// IsoSurfaceBlobsObject.h
// Definit la classe des objets "Blobs" c'est a dire composes de plusieurs blobs differents

#ifndef V3D_BLOBS_ISO_SURFACE_BLOBS_OBJECT_H_INCLUDED
#define V3D_BLOBS_ISO_SURFACE_BLOBS_OBJECT_H_INCLUDED

#include "IsoSurface.h"

#include "ScanLines.h"
#include "Basics/BoundAabb.h"

namespace V3D {

class BlobBase;
class Triangulator;


//////////////////////////////////////////////////////////////////////////////////////////

class IsoSurfaceBlobsObject : public IsoSurface
{
	typedef IsoSurface parent;

protected:
	// Association d'un intervalle d'indices de vertices avec un BoundAabb englobant les vertices correspondants
	typedef std::pair< ScanLineRange, BoundAabb> LocalisationInfo;
	typedef std::vector< LocalisationInfo > LocalisationInfoArray;

public:
	// Types de tableaux specifiques a cette classe
	typedef std::vector<const BlobBase*>   BlobPtrArray;

	// Pour chaque vertex, une valeur d'influence d'un blob
	typedef BlobBase::VertexInfluenceArray VertexInfluenceArray;
	// Liste des indices de vertex influences par un blob
	typedef BlobBase::VertexIndexArray     VertexIndexArray;

	// Tableau des influences pour tous les blobs (une entree par blob)
	typedef std::vector<VertexInfluenceArray> BlobsInfluencesValues;
	// Tableau des indices de vertex influences pour tous les blobs (une entree par blob)
	typedef std::vector<VertexIndexArray>     BlobsInfluencesIndices;
	
		//////////////////////////////////////////

public:
	explicit IsoSurfaceBlobsObject(float fEnergyThresh);
	virtual ~IsoSurfaceBlobsObject();

	// Accesseur a l'energie seuil de l'isosurface
    void SetEnergyThresh( float fThresh);
    float GetEnergyThresh() const { return m_fEnergyThreshold; }
	
	// Accesseur aux blobs de l'isosurface
	int32 GetActiveBlobsCount() const {return m_apBlobs.size(); }
	const BlobBase& GetActiveBlob( int32 nIndex ) const {return *(m_apBlobs[nIndex]); }

	// Ajoute un blob a la liste des blobs. (par reference car on veut s'assurer que le pointeur n'est pas null)
	// Attention : le blob n'est pas copie
	int32 AddBlob( const BlobBase& blob);
	
	// Reinitialisation de la liste des blobs de l'isosurface
	void RemoveAllBlobs();


	// Methode de creation du mesh en utilisant un Triangulator.
	// Plus efficace que d'appeler Triangulator::GenerateMesh 
	//         car ici les specificites de IsoSurfaceBlobsObject sont exploitees
	//         (par exemple les groupes de blobs independants sont traites separements...)
	void GenerateMesh( Triangulator& tessellator, 
                       Vect3DArray& aVertices, TriangleArray& aTriangles,
                       BlobsInfluencesIndices& blobsInfluences,
                       Vect3fArray* paNorms = 0, bool bNormalizeNormals = false ) const;


	// Calcul des valeurs d'influence des blobs sur les vertices influences
	void ComputeBlobsInfluences( BlobsInfluencesValues& vals, const BlobsInfluencesIndices& inflIdx, const Vect3DArray& aVtxBlobs ) const;

	// Determination, pour chaque blob, des vertices influences parmis les vertices du tableau fourni
	void FindBlobsInfluences( BlobsInfluencesIndices& infl, const Vect3DArray& aVtxBlobs,
	                          const LocalisationInfoArray* paVtxLocalisation = 0) const;

	// Determination, pour chaque blob, des vertices influences parmis les vertices du sous-tableau fourni
	void FindBlobsInfluencesSubArray( BlobsInfluencesIndices& infl, const Vect3DArray& aVtxBlobs,
	                                int32 nVertexStart, int32 nVertexCount,
	                                const LocalisationInfoArray* paVtxLocalisation = 0) const;


	// Combinaison de la determination des vertices influences 
	// et du calcul des normales a l'isosurface pour les positions du tableau fourni
	void ComputeSurfaceNormalsAndInfluencedVtx( Vect3fArray& avNorms, BlobsInfluencesIndices& infl, const Vect3DArray& aPos,
	                                            const LocalisationInfoArray* paVtxLocalisation = 0) const;

	// Combinaison de la determination des vertices influences 
	// et du calcul des normales a l'isosurface pour les positions du sous-tableau fourni
	void ComputeSurfaceNormalsAndInfluencedVtxSubArray( Vect3fArray& avNorms, BlobsInfluencesIndices& infl, const Vect3DArray& aPos,
	                                                  int32 nFirstIndex, int32 nNormalsToCompute,
	                                                  const LocalisationInfoArray* paVtxLocalisation = 0) const;


    
	// Calcul des normales a l'isosurface en chaque vertex du sous-tableau fourni
	virtual void ComputeSurfaceNormalsSubArray( Vect3fArray& avNorms, const Vect3DArray& aPos,
	                                          int32 nFirstIndex, int32 nNormalsToCompute) const;


	// Determination d'une boite englobante a l'isosurface
	virtual bool ComputeBounds( Vector3D& vcBoundMin, Vector3D& vcBoundMax ) const;


	// Determination de l'ensemble des cubes de la grille discrete de un cube d'epaisseur en Z
	// qui ne sont pas trivialement inutiles
	virtual bool FindInterestingPartsOfPlane( AllScansArray& aScansPlane,
	                                          const Vector3D& vcStart, const Vector3D& vcSteps,
	                                          int32 nStepsX, int32 nStepsY) const;
	
	// Valeur de la fonction de l'isosurface en un point ( val<0 : interieur du volume, val>0 : exterieur)
	virtual float FuncValueAtPos( const Vector3D& pos) const;

	// Valeur de la fonction de l'isosurface en une grille 3D de points
	virtual void FuncValuesForPlaneGrid( float* pafValues, 
	                            const Vector3D& vcStart, LengthType fDeltaX, LengthType fDeltaY,
	                            int32 nStepsX, int32 nStepsY,
	                            const AllScansArray* paScansCubesToClean = 0 ) const;


protected:

	// Classe interne utilisee pour determiner le voisinage des blobs entre eux
	// Les ensembles de blobs disjoints seront traites separements par grappes
	class SBlobsDirectNeighborhood;
	
	// Classe interne definissant une grappe de blobs (groupe de blobs qui se touchent)
	class SBlobsCluster;

protected:
	IsoSurfaceBlobsObject(const BlobPtrArray& apBlobs, float fEnergyThresh);

	// Calcul des valeurs pour une grille de points, connaissant optionnellement les cubes de la grille
	// qui sont "interessants", determines par la methode FindInterestingPartsOfPlane
	void ClearPlaneGridValues( float* pafValues, 
	                            int32 nPointsInLayerX, int32 nPointsInLayerY,
	                            const AllScansArray* paScansCubesToClean = 0) const;

	void BuildLocalisationInfo( LocalisationInfoArray& aLocInfo, 
	                            const Vect3DArray::const_iterator itVtxBeg,
	                            const Vect3DArray::const_iterator itVtxEnd,
	                            int32 nFirstVtxIdx ) const;

private:
	void AddSurfaceNormalsFindInfluenced( Vect3fArray& avNorms, BlobsInfluencesIndices& influenced,
	                                      const Vect3DArray& aPos,
	                                      const LocalisationInfoArray* paVtxLocalisation = 0) const;

	void AddSurfaceNormalsFindInfluencedSubArray( Vect3fArray& avNorms, BlobsInfluencesIndices& influenced,
                                                  const Vect3DArray& aPos,
                                                  int32 nStartIdx, int32 nCount,
	                                              const LocalisationInfoArray* paVtxLocalisation = 0) const;


private:
	BlobPtrArray	m_apBlobs;             // Liste des blobs visibles (Non proprietaire des blobs contenus)
	float			m_fEnergyThreshold;    // energie de seuil de l'isosurface
};



//////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////


// Classe definissant le voisinage des blobs : 
// pour chaque blob conserve la liste des blobs qui le touchent

class IsoSurfaceBlobsObject::SBlobsDirectNeighborhood
{
public:
	// Iterateur sur les indices de voisins d'un blob
	typedef std::vector<int32>::const_iterator NeighborIdxConstIter;

public:
	SBlobsDirectNeighborhood() {}

	inline int32 GetBlobsCount() const {return m_aDirectNeighborsInfos.size();}

	// Recuperation des iterateurs de debut et de fin des indices des voisins du blob d'index nBlobIdx
	inline void GetStartEndNeighborConstIters( NeighborIdxConstIter& start, 
	                                           NeighborIdxConstIter& end, int32 nBlobIdx) const
	         { 
				 const SBlobDirectNeighborhoodInfo& curBlobInfo = m_aDirectNeighborsInfos[nBlobIdx];
				 start = m_anDirectNeighborsIndices.begin() + curBlobInfo.nBlobNeighborsFirstIndex;
				 end   = start + curBlobInfo.nBlobNeighborsCount;
			 }

	// Construction des infos de voisinage des blobs (attention : goulot d'etranglement quand bcp de blobs)
	

	// Calcul tous les voisins de chaque blob ( N * (N-1) test d'intersection) :
	void ComputeDirectNeighborsAll( const BlobPtrArray& apBlobs );


	// ComputeDirectNeighborsAll <==> ComputeDirectNeighborsSuccessors + MakeDirectNeighborsSymetric

	// Calcul pour chaque blob de ses voisins situes apres lui dans la liste ( N * (N-1) / 2 test d'intersection) :
	void ComputeDirectNeighborsSuccessors( const BlobPtrArray& apBlobs );

	// Recuperation des predecesseurs
	void MakeDirectNeighborsSymetric();

private:
	// Intervalle du tableau de voisins concernant un blob
	struct SBlobDirectNeighborhoodInfo
	{
		int32 nBlobNeighborsFirstIndex;
		int32 nBlobNeighborsCount;
	};
	
	// Tableau d'intervalles du tableau de voisins concernant un blob (un element par blob)
	typedef  std::vector<SBlobDirectNeighborhoodInfo> SBlobDirectNeighborhoodInfoArray;

private:
	// On interdit la copie (trop couteux)
	SBlobsDirectNeighborhood(const SBlobsDirectNeighborhood&);
	SBlobsDirectNeighborhood& operator = (const SBlobsDirectNeighborhood&);

private:
	// Tableau lineaire des indices des voisins pour chaque blob
	std::vector<int32>                 m_anDirectNeighborsIndices;
	
	// Tableau des intervalles de m_anDirectNeighborsIndices correspondant a chaque blob
	SBlobDirectNeighborhoodInfoArray     m_aDirectNeighborsInfos;
};


//////////////////////////////////////////////////////////////////////////////////////////

// Classe decrivant les grappes de blobs

class IsoSurfaceBlobsObject::SBlobsCluster
{
public:
	// Construction a partir des infos de voisinage des blobs
	explicit SBlobsCluster(const SBlobsDirectNeighborhood& neighbInfo);
	
	// nombre de clusters
	inline int32 GetClustersCount() const {return m_anClustersFirstIdx.size(); }
	
	// Recuperation du nombre de blobs pour un cluster donne
	inline int32 GetBlobsCountInCluster(uint32 nClusterIdx) const
	{
		assert( nClusterIdx >= 0 && nClusterIdx < m_anClustersFirstIdx.size());
		if( nClusterIdx < int32(m_anClustersFirstIdx.size() - 1))
			return m_anClustersFirstIdx[nClusterIdx+1] - m_anClustersFirstIdx[nClusterIdx];
		else
			return m_anBlobsIdSortedByCluster.size() - m_anClustersFirstIdx[nClusterIdx];
	}

	// Recuperation du nBlobIdxInCluster ieme index de blob du cluster nClusterIdx
	inline const int32& GetBlobIdxInCluster(int32 nClusterIdx, int32 nBlobIdxInCluster) const
	{
		assert( nBlobIdxInCluster >= 0 && nBlobIdxInCluster < GetBlobsCountInCluster(nClusterIdx));
		int32 nFirstInCluster = m_anClustersFirstIdx[nClusterIdx];
		return m_anBlobsIdSortedByCluster[ nFirstInCluster + nBlobIdxInCluster ];
	}

private:
	// On interdit la copie (trop couteux)
	SBlobsCluster(const SBlobsCluster&);
	SBlobsCluster& operator = ( const SBlobsCluster&);

	// Methode pour la construction a partir des infos de voisinage
	void BuildFromNeighborhood( const SBlobsDirectNeighborhood& neighInfo);

private:
	// Correspondance Blob -> Identifiant du cluster auquel il appartient
	std::vector<int32> m_anBlobsClusterId;

	// Sequence des indices de blobs regroupes par cluster
	std::vector<int32> m_anBlobsIdSortedByCluster; 
	// Correspondance Blob -> index du premier element correspondant dans m_anBlobsIdSortedByCluster
	std::vector<int32> m_anClustersFirstIdx; 
};


//////////////////////////////////////////////////////////////////////////////////////////


} // namespace



#endif	// #ifndef V3D_BLOBS_ISO_SURFACE_BLOBS_OBJECT_H_INCLUDED

